<?php



/*

 * This file is part of the Symfony package.

 *

 * (c) Fabien Potencier <fabien@symfony.com>

 *

 * For the full copyright and license information, please view the LICENSE

 * file that was distributed with this source code.

 */



namespace Symfony\Component\Translation\Loader;



use Symfony\Component\Translation\Exception\InvalidResourceException;

use Symfony\Component\Translation\Exception\NotFoundResourceException;

use Symfony\Component\Config\Resource\FileResource;

use Symfony\Component\Yaml\Parser as YamlParser;

use Symfony\Component\Yaml\Exception\ParseException;



/**

 * YamlFileLoader loads translations from Yaml files.

 *

 * @author Fabien Potencier <fabien@symfony.com>

 *

 * @api

 */

class YamlFileLoader extends ArrayLoader implements LoaderInterface

{

    private $yamlParser;



    /**

     * {@inheritdoc}

     *

     * @api

     */

    public function load($resource, $locale, $domain = 'messages')

    {

        if (!stream_is_local($resource)) {

            throw new InvalidResourceException(sprintf('This is not a local file "%s".', $resource));

        }



        if (!file_exists($resource)) {

            throw new NotFoundResourceException(sprintf('File "%s" not found.', $resource));

        }



        if (null === $this->yamlParser) {

            $this->yamlParser = new YamlParser();

        }



        try {

            $messages = $this->yamlParser->parse(file_get_contents($resource));

        } catch (ParseException $e) {

            throw new InvalidResourceException('Error parsing YAML.', 0, $e);

        }



        // empty file

        if (null === $messages) {

            $messages = array();

        }



        // not an array

        if (!is_array($messages)) {

            throw new InvalidResourceException(sprintf('The file "%s" must contain a YAML array.', $resource));

        }



        $catalogue = parent::load($messages, $locale, $domain);

        $catalogue->addResource(new FileResource($resource));



        return $catalogue;

    }

}

